//___________________________________
//                                   \
// Camera Engine.  version 1.2	$Id: cinemathica.js 49 2007-05-16 23:45:50Z nilton $ fixed instantcam and personcam
//___________________________________/


/*
 Put Camera.UpdateAndDraw(); inside your Render script.

 Camera.active
	At all times, to check if the camera is active, you can check Camera.active
	Camera will automatically execute DetachCamera(), but will not reattach it when done.
 
 Camera.pan(X,Y,spdX,spdY)
	Will move your camera to pixels (X,Y) on your map with the set axis speed spdX and spdY.
	You can call it like: Camera.pan(X,Y,spdX) in which case it will use that speed for both axis.
	You can call it like Camera.pan(X,Y), which is the same as Camera.pan(X,Y,0), in this case,
	the Camera will repos instantly. Speeds must be >0.5 to work. For a good scroll, choose a speed
	that has your tile sizes as multiple. So 16 pixels tiles=> speed= 1,2,4,8,16,32,64.
	But thats just a recommendation.
	A shortwrite is available: PanCamXY()

 PanCamXY(X,Y,spdX,spdY)
	Will do the same as Camera.pan()

 PanCamTo(tileX,tileY,spdX,spdY)
	Same as PanCamXY(), but the coordinates you give are in tiles, not pixels.

 PanCamToPerson(sprite,spdX,spdY,attach)
	Same as above, but the destination is a sprite, and we have the option to attach the camera
	to that sprite when we're there.

 SlideCameraXY(x2,y2,spd)
	Ensures a linea-recta to the destination. The other functions will most probably bend the 
	direction a little at the end of the pan to get to the destination.

 To instantly complete a pan: Camera.disable(); 
 To Abort a pan mid-way: Camera.cancel();

 *** Tweaking ***
 If your TileWidth/Height isnt 16, you need to set: Camera.GTW=<number> and Camera.GTH=<number>
 do this before using any camera functions.


*/
/**
 * Create a Camera Object called 'Camera'.
 */
var Camera={active:false, x1:0,y1:0,x2:0,y2:0,x:0,y:0, spdX:0,spdY:0, dX:0,dY:0, GTW:16,GTH:16, abs:Math.abs,
	/**
	 * makes the camera pan to (x2,y2) with speeds (spdX, spdY)
	 * @param {integer} x2 Destination coordinate in Map pixels
	 * @param {integer} y2 Destination coordinate in Map pixels
	 * @param {float} spdX Panning speed, leave empty for instant pan
	 * @param {float} spdY If undefined, will be the same as spdX.
	 * It detaches the camera automatically, but will not attach it.
	 * Speeds need to be >0.5 
	 * example1: Camera.pan(10,10,0); //Instant new Camera Position (zero is optional)
	 * example2: Camera.pan(10,10,2); //Optional Y speed
	 */
	pan : function(x2, y2, spdX, spdY){
		this.x1 = this.x = GetCameraX();
		this.y1 = this.y = GetCameraY();
		if(IsCameraAttached())
			DetachCamera();
		this.UpdateAndDraw = this.run;
		this.active = true;
		this.x2 = x2;
		this.y2 = y2;
		if(!spdX)
			this.disable();
		if(!spdY)
			spdY = spdX;
		this.spdX = spdX;
		this.spdY = spdY;
		this.dX = this.x1<x2 ? spdX : -spdX;
		this.dY = this.y1<y2 ? spdY : -spdY;
	},
	/**
	 * Put this function inside your Render script:
	 * Camera.UpdateAndDraw();
	 */
	UpdateAndDraw: function(){},

        /**
         * You may want to use it to recalculate spdX/spdY/dX/dY values using posPercX and posPercY info.
	 * For example, to speed up in the middle and slowdown at the ends of a pan:
	 * Camera.recalc=function(){
	 * 	var x=this.posPercX()-50; //x now is between -50 and 50
	 * 	var y=this.posPercY()-50;
	 * 	var maxSpd = 5;  //Maximum speed in pixels per frame (will be reached at 50%, when x==0)
	 * 	var N = 50*50 / maxSpd;
	 * 	this.spdX= -x*x/N + maxSpd; //Set speed based on an inverted quadratic function
	 * 	this.spdY= -y*y/N + maxSpd;
	 * 	if(this.spdX<0.5) this.spdX=1;
	 * 	if(this.spdY<0.5) this.spdY=1;
	 * 	this.dX = this.x<this.x2 ? this.spdX : -this.spdX;
	 * 	this.dY = this.y<this.y2 ? this.spdY : -this.spdY;
	 * }
         */
	recalc: function(){},

	/**
	 * Redefine this function if you want something extra to be executed when the panning is done.
	 * @param {integer} d Has the value of 1 if disabled, undefined if the pan has been canceled
	 * example:
	 *  Camera.done=function(d){
	 * 	this.done=function(d){}; //It is for one time only, undefine this function
	 * 	if(d) {
	 * 		AttachInput(player2); //Now its player 2's turn.
	 * 	}else{
	 * 		AttachInput(player1);
	 * 		PanCamToPerson(player1,32,32,true); //Canceled! Back to player 1!
	 * 	}
	 *  }
	 *  DetachInput();PanCamToPerson(player2,8,8,true); //Now pan to player2, if we cancel() midway, we go back to player 1
	 */
	done: function(d){},

	/**
	 * Internal function called by Camera.UpdateAndDraw(), it will update the camera position.
	 */
	run: function(){
		this.x = GetCameraX();
		this.y = GetCameraY();
		this.recalc();
		if(this.x != this.x2){
			if(this.abs(this.x-this.x2) < this.spdX)
				this.x = this.x2;
			else
				this.x += this.dX;
		}
		if(this.y != this.y2){
			if(this.abs(this.y-this.y2) < this.spdY)
				this.y = this.y2; 
			else
				this.y += this.dY;
			}
		if(this.x==this.x2 && this.y==this.y2)
			this.disable();
		SetCameraX(this.x);
		SetCameraY(this.y);
	},
	/**
	 * Immediately cancel a running camera pan.
	 */
	cancel: function(d){
		this.UpdateAndDraw = function(){};
		this.active = false;
		this.done(d);
	},

	/**
	 * Instantly move Camera to the destination, then disable the camera pan.
	 */
	disable: function(){
		SetCameraX(this.x2);
		SetCameraY(this.y2);
		this.cancel(1);
	},
	/**
	 * Tile X coordinate to map X coordinate
	 * It will redefine itself with a faster function if the tilewidth is 16
	 * @param {number} xx A tile position on the map
	 * @returns a number in map coordinates
	 * that +7 is for the fact that to center on a tile, we need to be in the center of it, not in the upper corner.
	 */
	TileToMapX: function(xx) { 
		if(this.GTW==16)
			this.TileToMapX = new Function("xx", "{return (xx<<4)+7;}");
		else 
			this.TileToMapX = new Function("xx", "{return Math.floor(xx*this.GTW+this.GTW>>1-1);}");
		return Math.floor(xx*this.GTW+this.GTW>>1-1);
	},
	/**
	 * Tile Y coordinate to map Y coordinate
	 * It will redefine itself with a faster function if the tilewidth is 16
	 * @param {number} yy A tile position on the map
	 * @returns a number in map coordinates
	 */
	TileToMapY: function(yy) { 
		if(this.GTH==16)
			this.TileToMapY = new Function("yy", "{return (yy<<4)+7;}");
		else
			this.TileToMapY = new Function("yy", "{return Math.floor(yy*this.GTH+this.GTH>>1-1);}");
		return Math.floor(yy*this.GTH+this.GTH>>1-1);
	},

	/**
	 * Percentage panning done in the X axis.
	 * @param {integer} x Optional value in pixels, value must be between Camera.x1 and Camera.x2
	 * @returns a value beween 0 (start) and 100 (end), 50 is halfway.
	 * @type integer
	 */
	posPercX: function(x){ return (x||this.x-this.x1) * 100/(this.x2-this.x1); },

	/**
	 * Percentage panning done in the Y axis.
	 * @param {integer} y Optional value in pixels, value must be between Camera.y1 and Camera.y2
	 * @returns a value beween 0 (start) and 100 (end), 50 is halfway.
	 * @type integer
	 */
	posPercY: function(y){ return (y||this.y-this.y1) * 100/(this.y2-this.y1); },

	/**
	 * Set Zelda style screen-by-screen display, this way we have the illusion that a big map are chunks of smaller ones.
	 * unfortunately, we need to run this continously for it to work. Put it in your Update script.
	 */
	ReposCam: function(sprite){
		if(Camera.active)return;
		this.x = GetPersonX(sprite) - (GetPersonX(sprite)%GetScreenWidth())  + (GetScreenWidth()>>1);
		this.y = GetPersonY(sprite) - (GetPersonY(sprite)%GetScreenHeight()) + (GetScreenHeight()>>1);
		if( (GetCameraX() !=this.x) || (GetCameraY() !=this.y) )Camera.pan(this.x,this.y,24);
	}

}

/**
 * Makes the camera pan to (X,Y)  with speeds (spdX, spdY)
 * @param {integer} X Destination coordinate in Map pixels
 * @param {integer} Y Destination coordinate in Map pixels
 * @param {float} spdX Panning speed, leave empty for instant pan
 * @param {float} spdY If undefined, will be the same as spdX.
 * It detaches the camera automatically, but will not attach it.
 */
function PanCamXY(X,Y,spdX,spdY){Camera.pan(X,Y,spdX,spdY);}

/**
 * Makes the camera pan to (tileX,tileY) with speeds (spdX, spdY)
 * @param {integer} X Destination coordinate in Map Tiles
 * @param {integer} Y Destination coordinate in Map Tiles
 * @param {float} spdX Panning speed, leave empty for instant pan
 * @param {float} spdY If undefined, will be the same as spdX.
 * It detaches the camera automatically, but will not attach it.
 */
function PanCamTo(tileX,tileY,spdX,spdY){Camera.pan(Camera.TileToMapX(tileX),Camera.TileToMapY(tileY),spdX,spdY);}

/**
 * Ensures a linea-recta to the destination. The other functions will most probably bend the 
 * direction a little at the end of the pan to get to the destination.
 * @param {integer} X Destination coordinate in Map pixels
 * @param {integer} Y Destination coordinate in Map pixels
 * @param {float} spdX Panning speed, leave empty for instant pan
 * @param {float} spdY If undefined, will be the same as spdX.
 * It detaches the camera automatically, but will not attach it.
 */
function SlideCamXY(x2,y2,spd){
	var x1 = GetCameraX();
	var y1 = GetCameraY();
	if(x2 == x1)
		Camera.pan(x2,y2,spd); //avoid division by zero
	else 
		Camera.pan(x2,y2,spd,spd*(y2-y1)/(x2-x1)); // spdY=spd*m  (y=mx)
}

/**
 * Moves the camera to a person. Will work also on moving sprites :)
 * It will automatically detach the camera, if needed.
 * @param {string} sprite The name of the person to focus the camera to
 * @param {float} spdX Panning X speed  (if speed undefined, will instantly go to that person)
 * @param {float} spdY Panning Y speed 
 * @param {Boolean} attach Do we attach the camera when done? (defaults to false)
 */
function PanCamToPerson(sprite,spdX,spdY,attach){
	Camera.active=true;
	if(!spdX && !spdY){
		//Instant new Camera Position (zero is optional)
		Camera.x2=GetPersonX(sprite);
		Camera.y2=GetPersonY(sprite);
		Camera.disable();
		return;
	}
	Camera.x1 = Camera.x = GetCameraX();
	Camera.y1 = Camera.y = GetCameraY();
	if(spdY==undefined)spdY=spdX;     //Optional second speed
	Camera.spdX=spdX;
	Camera.spdY=spdY;

	if(IsCameraAttached())DetachCamera();
	Camera.UpdateAndDraw= function(){ 
		this.x2=GetPersonX(sprite);
		this.y2=GetPersonY(sprite);
		this.dX=GetCameraX()<this.x2?this.spdX:-this.spdX;
		this.dY=GetCameraY()<this.y2?this.spdY:-this.spdY;
		this.run();
		if(this.active)return;
		if(attach) AttachCamera(sprite);
	}
}

